import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'comprehensive.db');
    return new Database(dbPath);
}

// GET /api/tasks/[id]/recordings - Load recordings for a task
export async function GET(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    const db = getDb();
    try {
        const { id } = await params;
        const taskId = id; // Keep as string

        console.log('Loading recordings for task:', taskId);

        // Check if table exists first
        const tableExists = db.prepare(`SELECT name FROM sqlite_master WHERE type='table' AND name='task_recordings'`).get();
        if (!tableExists) {
            console.log('task_recordings table does not exist');
            return NextResponse.json([]);
        }

        const recordings = db.prepare(`
            SELECT id, name, duration, audio_data, created_at
            FROM task_recordings
            WHERE task_id = ?
            ORDER BY created_at DESC
        `).all(taskId);

        console.log('Found recordings:', recordings.length);
        return NextResponse.json(recordings);
    } catch (error: any) {
        console.error('Error loading recordings:', error);
        return NextResponse.json({ error: 'Failed to load recordings: ' + error.message }, { status: 500 });
    } finally {
        db.close();
    }
}

// POST /api/tasks/[id]/recordings - Save a new recording
export async function POST(
    request: NextRequest,
    { params }: { params: Promise<{ id: string }> }
) {
    const db = getDb();
    try {
        const { id } = await params;
        const taskId = id; // Keep as string
        console.log('💾 Saving recording for task:', taskId);

        const formData = await request.formData();

        const name = formData.get('name') as string;
        const duration = formData.get('duration') as string;
        const audioFile = formData.get('audio') as File;

        console.log('📋 Received data:', { name, duration, audioFile: audioFile?.name });

        if (!name || !duration || !audioFile) {
            console.error('❌ Missing required fields:', { name, duration, audioFile: !!audioFile });
            return NextResponse.json({ error: 'Missing required fields' }, { status: 400 });
        }

        const audioBuffer = Buffer.from(await audioFile.arrayBuffer());
        console.log('📦 Audio buffer size:', audioBuffer.length);

        // Check if table exists
        const tableExists = db.prepare(`SELECT name FROM sqlite_master WHERE type='table' AND name='task_recordings'`).get();
        if (!tableExists) {
            console.error('❌ task_recordings table does not exist');
            return NextResponse.json({ error: 'task_recordings table does not exist' }, { status: 500 });
        }

        const stmt = db.prepare(`
            INSERT INTO task_recordings (task_id, name, duration, audio_data)
            VALUES (?, ?, ?, ?)
        `);

        const result = stmt.run(taskId, name, duration, audioBuffer);
        console.log('✅ Recording saved with ID:', result.lastInsertRowid);

        return NextResponse.json({ id: result.lastInsertRowid });
    } catch (error: any) {
        console.error('Error saving recording:', error);
        return NextResponse.json({ error: 'Failed to save recording: ' + error.message }, { status: 500 });
    } finally {
        db.close();
    }
}
